#include "osapi.h"
#include "ets_sys.h"
#include "os_type.h"
#include "user_config.h"
#include "user_interface.h"
#include "gpio.h"
#include "mem.h"
#include "espconn.h"
#include "string.h"
#include "dwn.h"

#define TASK_QUEUE_SIZE 64
#define MAIN_LOOP_TASK_PRIORITY 1

// Valve failsafe = 30 minutes
#define FAILSAFE_INTERVAL (1000*60*30)
#define RELAY1_PERI PERIPHS_IO_MUX_MTDI_U
#define RELAY1_FUNC FUNC_GPIO12
#define RELAY1_PIN  BIT12
#define RELAY2_PERI PERIPHS_IO_MUX_MTCK_U
#define RELAY2_FUNC FUNC_GPIO13
#define RELAY2_PIN  BIT13
#define RELAY3_PERI PERIPHS_IO_MUX_MTMS_U
#define RELAY3_FUNC FUNC_GPIO14
#define RELAY3_PIN  BIT14
#define RELAY4_PERI PERIPHS_IO_MUX_GPIO2_U
#define RELAY4_FUNC FUNC_GPIO2
#define RELAY4_PIN  BIT2
#define LED_PERI PERIPHS_IO_MUX_GPIO0_U
#define LED_FUNC FUNC_GPIO0
#define LED_PIN  BIT0

static void main_loop(os_event_t *events);
os_event_t    main_loop_task_queue[TASK_QUEUE_SIZE];
static uint8 current_wifi_status = STATION_IDLE;
volatile os_timer_t failsafe;
volatile pwm_active = 0;
uint8 relays;

static void ICACHE_FLASH_ATTR turn_relay_off(uint8 relay);
static void ICACHE_FLASH_ATTR turn_relay_on(uint8 relay);

void failsafe_handler(void *arg)
{
    if (relays == 0) return;

    os_printf("failsafe handler. relays=%d\r\n",relays);
    turn_relay_off(0);

}

void ICACHE_FLASH_ATTR send_relay_status()
{
    char buf[2];
    buf[0] = 'R';
    buf[1] = relays;
    send_data(buf,2);
}

static void ICACHE_FLASH_ATTR client_recv_handler(struct espconn* client_connection, char* data, uint16 len)
{
    if (data ==0 | len <1) return;

    uint32 tmp;
    uint32 size = parse_number((char*)&data[1],len-1,&tmp);
    uint8 c = data[0]-'a'+1;
    if (c> 4) return;

    if (tmp == 0)
    {
        turn_relay_off(c);
    }
    else
    {
        turn_relay_on(c);
    }
}


static void ICACHE_FLASH_ATTR turn_relay_off(uint8 relay)
{
    if (relays == 0) return;

    // If relay != 0x0, check if the requested relay is currently on.
    // Otherwise it would mean we are attempting to turn off a relay
    // that is alreaydy off.
    //
    // If 0 is chosen, it means turn off all relays no matter what.

    if (relay != 0)
    {
        if (relays == 1 && relay!=1) return;
        if (relays == 2 && relay!=2) return;
        if (relays == 4 && relay!=3) return;
        if (relays == 8 && relay!=4) return;
    }

    gpio_output_set(RELAY1_PIN, 0, relay, 0);
    gpio_output_set(RELAY2_PIN, 0, relay, 0);
    gpio_output_set(RELAY3_PIN, 0, relay, 0);
    gpio_output_set(RELAY4_PIN, 0, relay, 0);

    os_timer_disarm(&failsafe);

    relays = 0;
    send_relay_status();
}

static void ICACHE_FLASH_ATTR turn_relay_on(uint8 relay)
{       
    if (relay > 4) return;
    if ((1<<(relay-1)) == relays) return;

    if (relay == 3)
    {
        gpio_output_set(RELAY2_PIN, 0, relay, 0);
        gpio_output_set(RELAY3_PIN, 0, relay, 0);
        gpio_output_set(RELAY4_PIN, 0, relay, 0);
        gpio_output_set(0, RELAY1_PIN, relay, 0);
    }
    else if (relay == 2)
    {
        gpio_output_set(RELAY1_PIN, 0, relay, 0);
        gpio_output_set(RELAY3_PIN, 0, relay, 0);
        gpio_output_set(RELAY4_PIN, 0, relay, 0);
        gpio_output_set(0, RELAY2_PIN, relay, 0);
    }
    else if (relay == 4)
    {
        gpio_output_set(RELAY1_PIN, 0, relay, 0);
        gpio_output_set(RELAY2_PIN, 0, relay, 0);
        gpio_output_set(RELAY4_PIN, 0, relay, 0);
        gpio_output_set(0, RELAY3_PIN, relay, 0);
    }
    else if (relay == 1)
    {
        gpio_output_set(RELAY1_PIN, 0, relay, 0);
        gpio_output_set(RELAY2_PIN, 0, relay, 0);
        gpio_output_set(RELAY3_PIN, 0, relay, 0);
        gpio_output_set(0, RELAY4_PIN, relay, 0);
    }

    os_timer_disarm(&failsafe);
    os_timer_arm(&failsafe, FAILSAFE_INTERVAL, 0);

    relays = 1<<(relay-1);
    send_relay_status();


}

static void ICACHE_FLASH_ATTR on_tcp_connection(struct espconn *new_connection) 
{
    char buf[2];
    buf[0] = 'S';
    buf[1] = relays;
    send_data(buf,2);
}

static void ICACHE_FLASH_ATTR update_wifi_status(uint8 status)
{
    current_wifi_status = status;
    if (status == STATION_GOT_IP)
    {
        gpio_output_set(0, LED_PIN, LED_PIN, 0);
    }
    else
    {
        gpio_output_set(LED_PIN, 0, LED_PIN, 0);
    }
}

static void main_loop(os_event_t *events)
{
    uint8 status = wifi_station_get_connect_status();
    if (status != current_wifi_status)
    {
        update_wifi_status(status);
        os_printf("wifi status: %d\r\n",status);
    }
    os_delay_us(10);
    system_os_post(MAIN_LOOP_TASK_PRIORITY, 0, 0);
} 


void ICACHE_FLASH_ATTR user_init()
{
    int i;

    system_set_os_print(1);
    uart_div_modify(0, UART_CLK_FREQ / 115200);

    // GPIO (needed for PWM)
    gpio_init();
    PIN_FUNC_SELECT(RELAY1_PERI, RELAY1_FUNC);
    PIN_FUNC_SELECT(RELAY2_PERI, RELAY2_FUNC);
    PIN_FUNC_SELECT(RELAY3_PERI, RELAY3_FUNC);
    PIN_FUNC_SELECT(RELAY4_PERI, RELAY4_FUNC);
    PIN_FUNC_SELECT(LED_PERI, LED_FUNC);

    // set all GPIO high. Relay will turn on when low.
    gpio_output_set(RELAY1_PIN, 0, RELAY1_PIN, 0);
    gpio_output_set(RELAY2_PIN, 0, RELAY2_PIN, 0);
    gpio_output_set(RELAY3_PIN, 0, RELAY3_PIN, 0);
    gpio_output_set(RELAY4_PIN, 0, RELAY4_PIN, 0);
    gpio_output_set(LED_PIN, 0, LED_PIN, 0);
    relays = 0;

    init_server(242,"ESP8266 IRRIGATION CONTROLLER",client_recv_handler,on_tcp_connection);

    os_timer_disarm(&failsafe);
    os_timer_setfn(&failsafe, (os_timer_func_t *)failsafe_handler, 0);

    // OS Task
    system_os_task(main_loop, MAIN_LOOP_TASK_PRIORITY, main_loop_task_queue, TASK_QUEUE_SIZE);
    system_os_post(MAIN_LOOP_TASK_PRIORITY, 0, 0);

}
