#include "JSON.h"
#include <string.h>
#include <sstream>
#include "Object.h"

using namespace Dumais::JSON;

/*
    A pure instance of a JSON object (i.e: not a inheriting Object, List or Value) has the sole purpose
    of holding a reference to an inheriting Object,List or Value. If it doesn't hold such a reference,
    it will ignore all operations, and will always return the value "{invalid}".

    When creating an instance of this object and then parsing a string, the mRoot will be a JSON::Object.
    All operations will be forwarded down to the JSON::Object. The Parsing operation will create
    all required entities and for this reason, this instance will be a root instance of a JSON tree. 
    Only the root instance should be allowed to delete what is in the tree

    Note that only the JSON::JSON object can create Object,List and Value objects. Therefore it is safe to 
    assume that deletion will be done from here at all times.

    When creating an instance of this object and using the operator= to assign it a value, the mRoot will reference
    the mRoot of the other JSON::JSON object. In that case, it means that this object is only a reference to another JSON entity
    and we should never delete the reference since this is not the root object.
*/

JSON::JSON(){
    mIsRoot = false;
    mRoot = this;
}

JSON::~JSON(){
    clean();
}

void JSON::clean()
{
    // this instance could be a root object that parsed and created sub-objects. But it could also
    // be a container that was assigned with the "=" operator. In that case, we don't want to delete
    // mRoot since the real container will do it
    if (!mIsRoot) return;
    // if mRoot = this instance, it means that this is a pure JSON instance with no root.
    if (mRoot != this)
    {
        delete mRoot;    
        mRoot = this;
    }
}

JSON::JSON(const JSON& json)
{
    mIsRoot = false;
    if (this != &json)
    {
        // only a pure JSON object can be re-assigned to another object
        assign(json);
    }
}

JSON& JSON::operator=(const JSON& json)
{
    if (this != &json)
    {
        // only a pure JSON object can be re-assigned to another object
        assign(json);
    }
    return *this;
}

void JSON::assign(const JSON& json)
{
    /* ===> WARNING <====
        We cannot use a JSON object as a return value of a function. This is because we
        are copying a reference to the root, whoch would exist of the stack of the called function.
        The solution would be to copy the whole content but then we could not dot obj2 = obj1["obj2"] because
        obj2 would be a copy and not a reference.
    
        There is no difference between doing "return obj1" and doing "obj2 = obj1".
        Both ways will call the copy constructor and in our case, the conpy constructor
        just copies a reference to the root object. 
    */
    clean();
    mRoot = json.mRoot;
    if (mRoot==0) mRoot = this;
}

JSON* JSON::copy()
{
    // if we get in here, it means we a re a pure instance
    return 0;
}

JSON& JSON::operator=(const std::string& json)
{
    setStringValue(json);
    return *this;
}


JSON& JSON::operator[](int i)
{
    return *mRoot->getByIndex(i);
}

JSON& JSON::operator[](std::string str)
{
    return *mRoot->getByKey(str);
}

std::string JSON::str()
{
    return mRoot->toString();
}

JSON* JSON::getByIndex(int i)
{
    return this;
}

JSON* JSON::getByKey(std::string key)
{
    return this;
}

std::string JSON::toString()
{
    return "{invalid}";
}

void JSON::parse(std::string json)
{
    //if we parse, then we are root. We should rememer that because this is the only instance that will be allowed to cleanup
    if (mRoot!=this) delete mRoot;

    // first pass: remove all spaces/tabs that are not part of strings
    Parser parser;
    json = mParser.removeSpaces(json);

    //root is always an object and never a list
    JSON *obj= new Object(json);
    if (obj)
    {
        mRoot = obj;
        mIsRoot = true;
    } else {
        mRoot = this;
        mIsRoot = false;
    }

}


std::string JSON::stringify(int level)
{
    return "{}";
}

std::string JSON::stringify(bool formatted)
{
    int level = -1;
    if (formatted) level = 1;

    return mRoot->stringify(level);
}

void JSON::setStringValue(std::string value)
{
}

JSON& JSON::addObject(const std::string& name)
{
    // if we get into this method, it means that a pure JSON was instanciated. if mRoot=this, it means it was not
    // even assigned yet.
    if (mRoot == this)
    {
        mRoot = new Object("{}");
        mIsRoot = true;
    }

    return mRoot->addObject(name);
}

JSON& JSON::addList(const std::string& name)
{
    // if we get into this method, it means that a pure JSON was instanciated. if mRoot=this, it means it was not
    // even assigned yet.
    if (mRoot == this)
    {
        mRoot = new Object("{}");
        mIsRoot = true;
    }

    return mRoot->addList(name);
}

JSON& JSON::addValue(const std::string& val,const std::string& name)
{
    // if we get into this method, it means that a pure JSON was instanciated. if mRoot=this, it means it was not
    // even assigned yet.
    if (mRoot == this)
    {
        mRoot = new Object("{}");
        mIsRoot = true;
    }

    return mRoot->addValue(val,name);
}

JSON& JSON::addValue(int val, const std::string& name)
{
    std::stringstream ss;
    ss << val;
    return addValue(ss.str(),name);
}

JSON& JSON::addValue(unsigned int val, const std::string& name)
{
    std::stringstream ss;
    ss << val;
    return addValue(ss.str(),name);
}


JSON& JSON::addValue(double val, const std::string& name)
{
    std::stringstream ss;
    ss << val;
    return addValue(ss.str(),name);
}

