#include "InsteonModem.h"
#include "../Logging.h"
#include <unistd.h>
#include "StandardOrExtendedMessage.h"
#include "AllLinkDatabaseMessage.h"
#include "IMConfigurationMessage.h"

InsteonModem::InsteonModem(char *serialPort, IInsteonMessageHandler *pHandler)
{
    pthread_mutex_init(&queueLock,0);
    mpSerialPort = new SerialPort(serialPort);
    mTimeLastSend = 0;
    mpInsteonMessageHandler = pHandler;

    // We will receive a 0x15 at the begining. Not sure why but just discard it.
    while (getByte()!=0x15);

    /* This sucks but it is the only way I could find to make the PLM
       pass the broadcast messages to the application. According to the 
       2412sdevguide.pdf, monitor mode is required because the PLM
        will only pass messages with the PLM's address to the application
    */
    setIMConfiguration(0b01000000); // set in Monitor mode
    getFirstAllLinkRecord();

}

InsteonModem::~InsteonModem()
{
    pthread_mutex_destroy(&queueLock);
    delete mpSerialPort;
    mpSerialPort = 0;
}


bool InsteonModem::process(bool readyToSend)
{
    // readyToSend is false if one of the device is expected to send an ACK. 
    // mWaitingForResponse is true if we sent a command to the PLM and we are waiting for a response from
    // the PLM (like get first ALL-link record etc...
    if (readyToSend && !mWaitingForResponse)
    {
        if (mInsteonCommandQueue.size()>0)
        {
            pthread_mutex_lock(&queueLock);
            IInsteonMessage* cmd = mInsteonCommandQueue.front();
            mInsteonCommandQueue.pop();
            pthread_mutex_unlock(&queueLock);
            sendCommand(cmd);
            delete cmd;
        }
    }

    int b = getByte();
    if (b>=0)
    {
        if (b==0x02)
        {
            waitForUnsolicitedMessage();
        }
    } else {
        return true;
    }

    return false;
}


int InsteonModem::getByte()
{
    //TODO: what is service is trying to stop? will we be stuck in here?
    char c;
    int size = mpSerialPort->Read(&c,1);
    if (size<1) return -1;

//    Logging::log("0x%x\r\n",c);
    return c; // make sure that bit 7 does not get transfered to bit 31 !!!
}

void InsteonModem::processAllLinkRecordResponse(unsigned char* buf)
{
    Logging::log("All-Link record response: %x%x%x flags=%x, group=%x",buf[4],buf[5],buf[6],buf[2],buf[3]);    
    mWaitingForResponse = false;
    getNextAllLinkRecord();
}


void InsteonModem::waitForUnsolicitedMessage()
{
    // Start of text was already read

    int index = 1;
    bool extended = false;
    unsigned char cmd;
    unsigned char cmd1,cmd2;
    InsteonID to,from;
    
    unsigned char buf[50];
    int b=-1;
    int length;
    while (b>=-1) // because -2 = stopping
    {
        b = getByte();
        if (b>=0)
        {
            if (index==1)
            {
                cmd = b;
                switch (cmd)
                {
                    case 0x50: length=11; break;
                    case 0x51: length=25; break;
                    case 0x52: length=4; break;
                    case 0x53: length=10; break;
                    case 0x54: length=3; break;
                    case 0x55: length=2; break;
                    case 0x56: length=7; break;
                    case 0x57: length=10; break;
                    case 0x58: length=3; break;
                }
            }
            buf[index]=b;
            index++;
            if (index==length)
            {
                switch (cmd)
                {
                    case 0x50: 
                        processStandardMessage((unsigned char*)&buf);            
                        break;
                    case 0x57:
                        processAllLinkRecordResponse((unsigned char*)&buf);
                        break;
                    case 0x51:
                    case 0x52:
                    case 0x53:
                    case 0x54:
                    case 0x55:
                    case 0x56:
                    case 0x58:
                    default:
                        Logging::log("Insteon unknown command received: %i",cmd);
                
                }
                break; // get out of loop
            }
        }
    }
}

bool InsteonModem::waitForReply(IInsteonMessage *cmd)
{
    int b = -1;
    int size=0;
    time_t t,t2;
    time(&t);
    while (b>=-1) // because -2 = stopping
    {
        b = getByte();
        time(&t2);
        
        if (b>=0)
        {
            // 2seconds have elapsed. Timeout
            if ((t2-t)>=2)
            {
                Logging::log("Timed out while waiting for Insteon echo");
                return false;
            }

            size++;
            if (size==cmd->getEchoSize())
            {
                if (b==0x06){
                    return true;
                } else if (b==0x15){
                    return false;
                } else {
                    Logging::log("Got unknown value while waiting for Insteon echo 0x%x",b);
                    return false;
                }
            }
        }
    }
    return false;
}

void InsteonModem::processStandardMessage(unsigned char *buf)
{
    InsteonID id = (buf[2]<<16)|(buf[3]<<8)|buf[4];
    Logging::log("Insteon cmd1=0x%x, cmd2=0x%x, flags=0x%x, device=0x%x",buf[9],buf[10],buf[8],id);
    mpInsteonMessageHandler->onInsteonMessage(buf);
}

bool InsteonModem::sendCommand(IInsteonMessage* cmd)
{

    std::string log = "Sending ";
    log+= cmd->toString();
    Logging::log(log.c_str());

    mpSerialPort->Write(cmd->getBuffer(),cmd->getSize());

    // We wait for the echo right away
    bool ack = waitForReply(cmd);
    if (!ack)
    {
        Logging::log("ERROR: got Nak from PLM");
        return false;
    }

    InsteonID id = cmd->getDestination();
    if (id==0 && cmd->needAck())
    {
        mWaitingForResponse = true;   
    }
    mpInsteonMessageHandler->onInsteonMessageSent(id,cmd);
    return true;
}

void InsteonModem::getFirstAllLinkRecord()
{

    AllLinkDatabaseMessage* cmd = new AllLinkDatabaseMessage(true);
    pthread_mutex_lock(&queueLock);
    mInsteonCommandQueue.push(cmd);
    pthread_mutex_unlock(&queueLock);
}

void InsteonModem::getNextAllLinkRecord()
{
    AllLinkDatabaseMessage* cmd = new AllLinkDatabaseMessage(false);

    pthread_mutex_lock(&queueLock);
    mInsteonCommandQueue.push(cmd);
    pthread_mutex_unlock(&queueLock);
}

void InsteonModem::setIMConfiguration(unsigned char imcmd)
{
    IMConfigurationMessage* cmd = new IMConfigurationMessage(imcmd);

    pthread_mutex_lock(&queueLock);
    mInsteonCommandQueue.push(cmd);
    pthread_mutex_unlock(&queueLock);

}

void InsteonModem::writeCommand(InsteonID destination, unsigned char cmd1, unsigned char cmd2)
{
    StandardOrExtendedMessage *cmd = new StandardOrExtendedMessage(destination,cmd1,cmd2);
    pthread_mutex_lock(&queueLock);
    mInsteonCommandQueue.push(cmd);
    pthread_mutex_unlock(&queueLock);
}

void InsteonModem::writeExtendedCommand(InsteonID destination, unsigned char cmd1, unsigned char cmd2, unsigned char* data)
{
    StandardOrExtendedMessage *cmd = new StandardOrExtendedMessage(destination,cmd1,cmd2, data);
    pthread_mutex_lock(&queueLock);
    mInsteonCommandQueue.push(cmd);
    pthread_mutex_unlock(&queueLock);

}
