#include "Logging.h"
#include "WeatherHelper.h"
#include <cmath>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string>
#include <stdio.h>
#include <netinet/tcp.h>
#include <sstream>
#include <stdlib.h>
#include <string.h>
#include "config.h"
#include "json/JSON.h"

#define JAN12000 946684800.0
#define JULIANJAN12000 2451545.0
#define DEG (M_PI/180.0)
#define GMT_CORRECTION (5*60*60)

WeatherHelper::WeatherHelper()
{
    mLastDay=0;
    mLongitude = LONGITUDE;
    mLatitude = LATITUDE;
    mLastTemperatureCheck=0;

}

WeatherHelper::~WeatherHelper()
{
}

void WeatherHelper::recalculate()
{
    time_t t;
    double julianDate;
    time(&t);

    // TODO: the mLastDay skips over a day if calculated after midnight UTC
    mLastDay = (floor((double)t/(double)((24*60*60)))+1)*(24*60*60)+GMT_CORRECTION; // tomorow midnight
    Logging::log("Recalculating sunset/sunrise. Next recalculation will occur at %i",mLastDay);
    t = (t-JAN12000)/(60*60*24);
    julianDate = t  + JULIANJAN12000;

    //TODO: should use all radians instead of converting
    double julianCycle = round(julianDate - JULIANJAN12000 -0.0009 +(mLongitude/360.0));
    double solarNoon = JULIANJAN12000 + julianCycle + 0.0009-(mLongitude/360.0);
    double M = fmod((357.5291 + 0.98560028 * (solarNoon-JULIANJAN12000)),360.0);
    double C = (1.9148 * sin(DEG*M)) + (0.0200 * sin(DEG*2 * M)) + (0.0003 * sin(DEG*3 * M)) ;
    double lambda = fmod((M + 102.9372 + C + 180.0),360);
    double jt = solarNoon + (0.0053 * sin(DEG*M)) - (0.0069 * sin(DEG*2.0 * lambda));
    double teta = asin(sin(DEG*lambda)*sin(DEG*23.45))/DEG;

    double H = acos((sin(DEG*(-0.83)) - sin(DEG*mLatitude) * sin(DEG*teta)) / (cos(DEG*mLatitude) * cos(DEG*teta)))/DEG;
    double aprox = JULIANJAN12000 + 0.0009 + ((H - mLongitude)/360.0) + julianCycle;
    double sunset = aprox + (0.0053 * sin(DEG*M)) - (0.0069 * sin(DEG*2.0 * lambda));
    double sunrise = jt-(sunset-jt);

    double zoneCorrection = (12)*3600;
    sunset = ((sunset - JULIANJAN12000)*86400.0)+JAN12000 + zoneCorrection;
    sunrise = ((sunrise - JULIANJAN12000)*86400.0)+JAN12000 + zoneCorrection;

    mSunset = sunset;
    mSunrise = sunrise;
    

    Logging::log("Sunrise: %i, Sunset: %i\r\n",mSunrise, mSunset);
}

time_t WeatherHelper::getSunRise()
{
    time_t t;
    time(&t);
    if (t>mLastDay) recalculate();
    
    return mSunrise;
}

time_t WeatherHelper::getSunSet()
{
    time_t t;
    time(&t);
    if (t>mLastDay) recalculate();
    
    return mSunset;

}


int WeatherHelper::getCurrentTemperature()
{
    // For now we get temperature from a web API. But it would be nice to
    // get a real thermomether and get it from there.

    // We will cache the last reading for 15min
    time_t t;
    time(&t);
    if (t < (mLastTemperatureCheck+(60*15)))
    {
        return mCurrentTemperature;
    }
    
    std::stringstream ss;
    ss <<"GET /feed/weather.ashx?q="<<mLatitude<<","<<mLongitude;
    ss<<"&format=json&num_of_days=1&key=ee77bf3145001848132301 HTTP:/1.1\r\n";
    ss<<"Host: free.worldweatheronline.com\r\n\r\n";

    sockaddr_in sockadd;
    int sock=socket(AF_INET,SOCK_STREAM,0);
    memset(&sockadd,0,sizeof(sockadd));
    struct hostent *he;
    he = gethostbyname("free.worldweatheronline.com");
    sockadd.sin_family = AF_INET;
    sockadd.sin_port = htons(80);
    memcpy(&sockadd.sin_addr, he->h_addr_list[0], he->h_length);
    Dumais::JSON::JSON json;
    if (connect(sock,(struct sockaddr *)&sockadd, sizeof(sockadd)) >= 0)
    {
        send(sock,ss.str().c_str(),ss.str().size(),0);
        char buf[1024];
        std::string st;
        int size;
        while ((size=recv(sock,(char*)&buf,1024-1,0))>0)
        {
            buf[size]=0;
            st += (char*)&buf;
        }
        st = st.substr(st.find("\r\n\r\n")+4,std::string::npos);
        json.parse(st);
        close(sock);
    }

    mCurrentTemperature=-100;
    std::string temp = json["data"]["current_condition"][0]["temp_C"].str();
    if (temp!="{invalid}")
    {
        mCurrentTemperature = atoi(temp.c_str());
        mLastTemperatureCheck=t;
    }

    return mCurrentTemperature;
}

