#include "Logging.h"
#include "RTPSession.h"
#include "SoundFile.h"
#include "SilenceSound.h"
#include <sstream>

#define CONF_RTP_PAYLOAD_SIZE 160

void *threadStarter(void *p)
{
    RTPSession *f = (RTPSession*)p;
    f->run();
}


RTPSession::RTPSession()
{
    this->mSession=rtp_session_new(RTP_SESSION_SENDRECV);
    this->mLocalIP = "";
    this->mLocalPort=0;
    pthread_mutex_init(&this->mQueueLock,0);
    this->mAbort = true;
    this->mMusic = 0;
}

RTPSession::~RTPSession(){
    this->stop();

    // Clear the queue
    while(this->mSoundQueue.size())
    {
        ISound *s = this->mSoundQueue.front();
        delete s;
        this->mSoundQueue.pop();
    }

    pthread_mutex_destroy(&this->mQueueLock);

}


void RTPSession::start()
{
    this->mAbort = false;

    rtp_session_set_local_addr(mSession,this->mLocalIP.c_str(),this->mLocalPort);
    rtp_session_set_scheduling_mode(mSession,1);
    rtp_session_set_blocking_mode(mSession,1);
    rtp_session_set_connected_mode(mSession,TRUE);
    rtp_session_set_remote_addr(mSession,this->mPeerIP.c_str(),this->mPeerPort);
    rtp_session_set_payload_type(mSession,0);

    //TODO: use thread pool
    pthread_create(&this->mThreadHandle, 0, threadStarter, (void*)this);

}

void RTPSession::stop()
{

    if (!this->mAbort && this->mSession!=0)
    {
        this->mAbort = true;
        int ret = pthread_join(this->mThreadHandle,0);
        rtp_session_destroy(this->mSession);
        this->mSession = 0;
    }
}

void RTPSession::silence(int seconds)
{
    SilenceSound *sound = new SilenceSound(seconds);
    pthread_mutex_lock(&this->mQueueLock);
    this->mSoundQueue.push(sound);
    pthread_mutex_unlock(&this->mQueueLock);
}

void RTPSession::play(const char *filename)
{
    SoundFile *file = new SoundFile();
    if (file->open(filename,G711))
    {
        pthread_mutex_lock(&this->mQueueLock);
        this->mSoundQueue.push(file);        
        pthread_mutex_unlock(&this->mQueueLock);
    } else {
        delete file;
    }
}

void RTPSession::setMusic(const char *filename)
{
    if (this->mMusic) delete this->mMusic;
    this->mMusic = new SoundFile(true);
    ((SoundFile*)(this->mMusic))->open(filename,G711);
}


//WARNING: currently only support 0-9
void RTPSession::playNumber(int number)
{
/*    char c[6];
    c[0] = number+48;
    c[1] = '.';
    c[2] = 'w';
    c[3] = 'a';
    c[4] = 'v';
    c[5] = 0;
    std::stringstream ss;
    ss << SOUND_PATH << (char*)&c;
    this->play(ss.str().c_str());*/

    
}

void RTPSession::setPeerAddress(std::string ip, unsigned int peerPort)
{
    this->mPeerIP = ip;
    this->mPeerPort= peerPort;
}

void RTPSession::setLocalAddress(std::string ip, unsigned int port)
{
    this->mLocalPort = port;
    this->mLocalIP = ip;
}

unsigned int RTPSession::getLocalPort()
{
    return this->mLocalPort;
}

std::string RTPSession::getLocalIP()
{
    return this->mLocalIP;
}


void RTPSession::clearQueue()
{
    pthread_mutex_lock(&this->mQueueLock);
    this->mSoundQueue=std::queue<ISound*>();
    this->mInterruptCurrentSound = true;
    pthread_mutex_unlock(&this->mQueueLock);
}

void RTPSession::run()
{
    unsigned int user_ts;
    unsigned char buffer[CONF_RTP_PAYLOAD_SIZE];

    ISound *currentSound=0;
    bool queueWasEmpty=true;
    this->mInterruptCurrentSound=false;
    bool resetTime = false;
    while (!this->mAbort)
    {
        if (!currentSound)
        {
            pthread_mutex_lock(&this->mQueueLock);
            if (this->mSoundQueue.size())
            {
                currentSound = this->mSoundQueue.front();
                this->mSoundQueue.pop();
                pthread_mutex_unlock(&this->mQueueLock);
                queueWasEmpty=false;
                this->mInterruptCurrentSound=false;
            } else {
                pthread_mutex_unlock(&this->mQueueLock);
                if (!queueWasEmpty)
                {
                    this->notifyQueueEmpty();
                    queueWasEmpty=true;
                }
                usleep(100000);
            }
        }


        int soundindex = 0;
        bool silenceSuppresion = true;
        if (this->mMusic)
        {
            this->mMusic->getSample(CONF_RTP_PAYLOAD_SIZE,(char*)&buffer);
            silenceSuppresion=false;
        }

        if (currentSound)   // if there is a sound to be played from queue, hide the music
        {
            silenceSuppresion= false;
            // overwrite music with sound
            soundindex=currentSound->getSample(CONF_RTP_PAYLOAD_SIZE,(char*)&buffer);
            //Note: if buffer wasn't filled completely, it contains music anyways.

            if (soundindex<CONF_RTP_PAYLOAD_SIZE || this->mInterruptCurrentSound)
            {
                delete currentSound;
                currentSound = 0;
            }
        }

        if (!silenceSuppresion)
        {
            if (resetTime)
            {
                user_ts = rtp_session_get_current_send_ts(this->mSession);
                resetTime = false;
            }
            rtp_session_send_with_ts(this->mSession,(uint8_t*)&buffer,CONF_RTP_PAYLOAD_SIZE,user_ts);
            user_ts+=CONF_RTP_PAYLOAD_SIZE;
        } else {
            // we started skipping frames, so we will need to resync next time
            resetTime = true;
        }

    }

}

void RTPSession::addObserver(IRTPSessionObserver* obs)
{
    this->mObservers.push_back(obs);
}

int RTPSession::getSoundQueueSize()
{
    int size = 0;
    pthread_mutex_lock(&this->mQueueLock);
    size = this->mSoundQueue.size();
    pthread_mutex_unlock(&this->mQueueLock);
    return size;
}

void RTPSession::notifyQueueEmpty()
{
    for (std::list<IRTPSessionObserver*>::iterator it = this->mObservers.begin();it!=this->mObservers.end();it++)
    {
        (*it)->onSoundQueueEmpty();
    }
}
