#include "Logging.h"
#include "PhoneService.h"
#include "config.h"
#include "SoundListParser.h"
#include <resip/dum/ClientSubscription.hxx>

using namespace resip;

PhoneService::PhoneService()
{
    mMWI = 0;
    mSIPEngine.setTelephonyObserver(this);

    this->addCallBack("call",new ServiceCallBack<PhoneService>(this,&PhoneService::call_callback));
    this->addCallBack("register",new ServiceCallBack<PhoneService>(this,&PhoneService::register_callback));
    this->addCallBack("blf",new ServiceCallBack<PhoneService>(this,&PhoneService::blf_callback));
    this->addCallBack("showcalls",new ServiceCallBack<PhoneService>(this,&PhoneService::showcalls_callback));
    this->addCallBack("showblf",new ServiceCallBack<PhoneService>(this,&PhoneService::showblf_callback));
    this->addCallBack("release",new ServiceCallBack<PhoneService>(this,&PhoneService::release_callback));
    this->addCallBack("play",new ServiceCallBack<PhoneService>(this,&PhoneService::play_callback));

}

PhoneService::~PhoneService()
{
}


void PhoneService::call_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    bool releaseAfterSounds = true;
    if (params["releaseaftersounds"].str()=="false") releaseAfterSounds = false;
    this->call(params["ext"].str(),params["play"].str(),releaseAfterSounds);

}

void PhoneService::register_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    this->registerUserAgent(params["user"].str(),params["pin"].str(),params["proxy"].str());

}

void PhoneService::blf_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    this->subscribeBLF(params["ext"].str());
}

void PhoneService::showcalls_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    this->getCallsList(json);
}

void PhoneService::showblf_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    this->getBLFList(json);
}

void PhoneService::release_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    this->releaseCall(params["id"].str());
}

void PhoneService::play_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    bool releaseAfterSounds = true;
    if (params["releaseaftersounds"].str()=="false") releaseAfterSounds = false;
    this->playOnCall(params["id"].str(),params["sound"].str(),releaseAfterSounds);
}

void PhoneService::onMWI(int num)
{
    mMWI = num;
}

void PhoneService::onAnswer(Call *pCall)
{
    SoundListParser parser(pCall->mPlayString);
    std::vector<std::string> list = parser.getSoundList();
    for (std::vector<std::string>::iterator it = list.begin();it!=list.end();it++)
    {
        std::string sound = *it;
        if (sound[0]=='$')
        {
            std::string st = sound.substr(1,std::string::npos);
            int duration = atoi(st.c_str());
            Logging::log("Queuing %i seconds silence on call \r\n",duration);
            pCall->getRTPSession()->silence(duration);
        } else {
            std::string st = SOUNDS_FOLDER+(sound)+"-ulaw.wav";
            Logging::log("Queuing %s on call \r\n",st.c_str());
            pCall->getRTPSession()->play(st.c_str());
        }
    }
}

void PhoneService::onNewDevicePresence(Subscription *pSub)
{
    //Logging::log("PhoneService::onNewDevicePresence: %i",pSub);

    // After expiry, we will get here but it will still be the same Subscription instance
    if (mBLFList.find(pSub)==mBLFList.end())
    {
        // it is not in the list. Add it.
        mBLFList[pSub]="unknown";
    }
}

int PhoneService::getMWI()
{
    return mMWI;
}

void PhoneService::onPresence(Subscription *pSub)
{
    BLFList::iterator it = mBLFList.find(pSub);
    if (it!=mBLFList.end())
    {
        if (pSub->getDeviceState()!=it->second) // dont send event if status is the same
        {
            mBLFList[pSub] = pSub->getDeviceState(); 
            Dumais::JSON::JSON json;
            json.addValue("presence","event");
            json.addValue(pSub->getDevice(),"device");
            json.addValue(pSub->getDeviceState(),"status");
            mpEventProcessor->processEvent(json);

            Logging::log("Presence: device %s state is %s",pSub->getDevice().c_str(),pSub->getDeviceState().c_str());

            if (pSub->getDeviceState()=="terminated")
            {
                mBLFList.erase(it);
            }
        }
    }
}


void PhoneService::onCallTerminated(Call *call)
{
    std::map<std::string,Call*>::iterator it = mCallsList.find(call->getID());
    if (it!=mCallsList.end()) mCallsList.erase(it);

    Dumais::JSON::JSON json;
    json.addValue("call","event");
    json.addObject("call");
    json["call"].addValue(call->mFrom.uri().user().data(),"from");
    json["call"].addValue(call->mTo.uri().user().data(),"to");
    json["call"].addValue(call->getID(),"id");
    json.addObject("callevent");
    json["callevent"].addValue("released","type");
    json["callevent"].addValue(call->isIncomming()?"incoming":"outgoing","dir");
    mpEventProcessor->processEvent(json);
}


bool PhoneService::onNewCallUas(Call *call)
{
    Logging::log("Incoming call\r\n");
    for (std::map<std::string,Call*>::iterator it=mCallsList.begin();it!=mCallsList.end();it++)
    {
        // Only allow one incomming call at a atime
        if (it->second->isIncomming())
        {
            return false;
        }
    }
    call->addRTPObserver(this);
    mCallsList[call->getID()]=call;
    return true;
}



bool PhoneService::onNewCallUac(Call *call)
{
    mCallsList[call->getID()]=call;
    return true;
}

void PhoneService::onDigit(Call *call, std::string digit)
{
    // when we press a digit, we must clear the sound queue
    call->getRTPSession()->clearQueue();

    Dumais::JSON::JSON json;
    json.addValue("call","event");
    json.addObject("call");
    json["call"].addValue(call->mFrom.uri().user().data(),"from");
    json["call"].addValue(call->mTo.uri().user().data(),"to");
    json["call"].addValue(call->getID(),"id");
    json.addObject("callevent");
    json["callevent"].addValue("digit","type");
    json["callevent"].addValue(digit,"digit");
    mpEventProcessor->processEvent(json);


}

void PhoneService::onConnectedUas(Call *call)
{
    Dumais::JSON::JSON json;
    json.addValue("call","event");
    json.addObject("call");
    json["call"].addValue(call->mFrom.uri().user().data(),"from");
    json["call"].addValue(call->mTo.uri().user().data(),"to");
    json["call"].addValue(call->getID(),"id");
    json.addObject("callevent");
    json["callevent"].addValue("answered","type");
    json["callevent"].addValue("incoming","dir");
    mpEventProcessor->processEvent(json);
}

void PhoneService::subscribeBLF(std::string device)
{
    mSIPEngine.subscribeBLF(device);
}

void PhoneService::subscribeMWI(std::string device)
{
    mSIPEngine.subscribeMWI(device);
}

void PhoneService::getCallsList(Dumais::JSON::JSON& json)
{
    //TODO: This will be called from another thread. Make it safe
    json.addList("calls");
    for (std::map<std::string,Call*>::iterator it = mCallsList.begin();it!=mCallsList.end();it++)
    {
        Call *pCall = it->second;
        pCall->toJSON(json["calls"]);
    }

}

void PhoneService::getBLFList(Dumais::JSON::JSON& json)
{
    //TODO: This will be called from another thread. Make it safe
    json.addList("subscriptions");
    for (BLFList::iterator it = mBLFList.begin();it!=mBLFList.end();it++)
    {
        Subscription *pSub = it->first;
        pSub->toJSON(json["subscriptions"]);
    }

}


void PhoneService::releaseCall(std::string id)
{
    //TODO: This will be called from another thread. Make it safe

    std::map<std::string,Call*>::iterator it = mCallsList.find(id);
    if (it!=mCallsList.end())
    {
        mSIPEngine.releaseCall(it->second);
    }
}

void PhoneService::playOnCall(std::string id, std::string playstring, bool hangupAfterSounds)
{
    //TODO: This will be called from another thread. Make it safe
    std::map<std::string,Call*>::iterator it = mCallsList.find(id);
    if (it!=mCallsList.end())
    {
        Call *pCall = it->second;
        pCall->mHangupAfterSounds = hangupAfterSounds;
        SoundListParser parser(playstring);
        std::vector<std::string> list = parser.getSoundList();
        for (std::vector<std::string>::iterator it = list.begin();it!=list.end();it++)
        {
            std::string sound = *it;
            if (sound[0]=='$')
            {
                std::string st = sound.substr(1,std::string::npos);
                int duration = atoi(st.c_str());
                Logging::log("Queuing %i seconds silence on call \r\n",duration);
                pCall->getRTPSession()->silence(duration);
            } else {
                std::string st = SOUNDS_FOLDER+(*it)+"-ulaw.wav";
                Logging::log("Queuing %s on call \r\n",st.c_str());
                pCall->getRTPSession()->play(st.c_str());
            }
        }
    }

}

Call* PhoneService::call(std::string destination,std::string playString, bool hangupAfterSounds)
{

    //TODO: This will be called from another thread. Make it safe
    Call *call = mSIPEngine.makeCall(destination);
    if (call==0) return 0;

    call->mHangupAfterSounds = hangupAfterSounds;
    call->setPlayString(playString);
    call->addRTPObserver(this);

    return call;
}




void PhoneService::run()
{
    while (!stopping())
    {
        mSIPEngine.run();
    }
}

void PhoneService::registerUserAgent(std::string user, std::string pin, std::string proxy)
{
    //TODO: This will be called from another thread. Make it safe
    AccountSettings settings;
    settings.mUserName = user;
    settings.mPin = pin;
    settings.mProxy = proxy;
    settings.mDisplayName = user;

    mSIPEngine.registerUserAgent(settings);

}


void PhoneService::onRTPSessionSoundQueueEmpty(Call *call)
{
    if (call->mHangupAfterSounds)
    {
        mSIPEngine.releaseCall(call);
    }
}



