#include "Logging.h"
#include "Insteon.h"
#include "insteon/SwitchDevice.h"
#include "insteon/EZFloraDevice.h"
#include <stdio.h>
#include <stdlib.h>
#include <iomanip>
#include "insteon/StandardOrExtendedMessage.h"

#define MAX_READ_SIZE 1024


Insteon::Insteon(char *serialPort)
{
    this->addCallBack("clearmodules",new ServiceCallBack<Insteon>(this,&Insteon::clearmodules_callback));
    this->addCallBack("listmodules",new ServiceCallBack<Insteon>(this,&Insteon::listmodules_callback));
    this->addCallBack("addmodule",new ServiceCallBack<Insteon>(this,&Insteon::addmodule_callback));
    this->addCallBack("addezflora",new ServiceCallBack<Insteon>(this,&Insteon::addezflora_callback));
    this->addCallBack("setcontroller",new ServiceCallBack<Insteon>(this,&Insteon::setcontroller_callback));
    this->addCallBack("ramp",new ServiceCallBack<Insteon>(this,&Insteon::ramp_callback));
    this->addCallBack("switch",new ServiceCallBack<Insteon>(this,&Insteon::switch_callback));
    this->addCallBack("ezflora/setprogram",new ServiceCallBack<Insteon>(this,&Insteon::ezflorasetprogram_callback));
    this->addCallBack("ezflora/startprogram",new ServiceCallBack<Insteon>(this,&Insteon::ezflorastartprogram_callback));
    this->addCallBack("ezflora/stopprogram",new ServiceCallBack<Insteon>(this,&Insteon::ezflorastopprogram_callback));
    this->addCallBack("ezflora/startvalve",new ServiceCallBack<Insteon>(this,&Insteon::ezflorastartvalve_callback));
    this->addCallBack("ezflora/stopvalve",new ServiceCallBack<Insteon>(this,&Insteon::ezflorastopvalve_callback));
    this->addCallBack("ezflora/status",new ServiceCallBack<Insteon>(this,&Insteon::ezfloraforcegetvalvestatus_callback));
    this->addCallBack("raw",new ServiceCallBack<Insteon>(this,&Insteon::raw_callback));
    mpInsteonModem = new InsteonModem(serialPort,this);
}

Insteon::~Insteon()
{
    delete mpInsteonModem;
}


void Insteon::run()
{
    while (!stopping())
    {
        // Make sure every devices are idle before atempting to send another command
        bool devicesIdle = true;
        for (std::map<InsteonID,InsteonDevice*>::iterator it = mModules.begin();it!=mModules.end();it++)
        {
            time_t t;
            time(&t);
            it->second->onTimer(t);
            if (!it->second->isIdle())
            {
                devicesIdle = false;
                break;
            }
        }

        if (mpInsteonModem->process(devicesIdle))
        {
            // dont sleep more than 50ms because Insteon modem expects reponses within 85ms
            usleep(50000);
        }

    }
}

void Insteon::clearmodules_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    clearModuleList();
    json.addValue("ok","status");
}

void Insteon::listmodules_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    json.addList("modules");
    listModules(json["modules"]);
}

void Insteon::addmodule_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    if (params["id"].str()!="{invalid}" && params["name"].str()!="{invalid}")
    {
        InsteonID id = strtoul(params["id"].str().c_str(),0,16);
        addModuleDefinition(id,params["name"].str());
        json.addValue("ok","status");
    }
}

void Insteon::addezflora_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    if (params["id"].str()!="{invalid}" && params["name"].str()!="{invalid}")
    {
        InsteonID id = strtoul(params["id"].str().c_str(),0,16);
        addEZFlora(id,params["name"].str());
        json.addValue("ok","status");
    }
}

void Insteon::setcontroller_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    if (params["id"].str()!="{invalid}")
    {
        InsteonID id = strtoul(params["id"].str().c_str(),0,16);
        setController(id);
        json.addValue("ok","status");
    }
}

void Insteon::ramp_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    unsigned char rate = 0x0F;
    unsigned char level = 0x0F;
    if (params["rate"].str()!="{invalid}") rate = atoi(params["rate"].str().c_str());
    if (params["level"].str()!="{invalid}") level = atoi(params["level"].str().c_str());
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    if (params["action"].str()=="on"){
        rampOn(id,level,rate);
    } else if (params["action"].str()=="off"){
        rampOff(id,rate);
    }
}

void Insteon::switch_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    if (params["action"].str()=="on"){
        unsigned char level =255;
        if (params["level"].str()!="{invalid}") level = atoi(params["level"].str().c_str());
        if (params["rate"].str()!="{invalid}") level = atoi(params["rate"].str().c_str());
        lightOn(id,level);
    } else if (params["action"].str()=="off"){
        lightOff(id);
    }
    json.addValue("ok","status");
}

void Insteon::ezflorasetprogram_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned char program = atoi(params["p"].str().c_str());
    if (program>=0 && program<=4)
    {
        unsigned char z1 = (params["z1"].str()=="{invalid}")?0:atoi(params["z1"].str().c_str());
        unsigned char z2 = (params["z2"].str()=="{invalid}")?0:atoi(params["z2"].str().c_str());
        unsigned char z3 = (params["z3"].str()=="{invalid}")?0:atoi(params["z3"].str().c_str());
        unsigned char z4 = (params["z4"].str()=="{invalid}")?0:atoi(params["z4"].str().c_str());
        unsigned char z5 = (params["z5"].str()=="{invalid}")?0:atoi(params["z5"].str().c_str());
        unsigned char z6 = (params["z6"].str()=="{invalid}")?0:atoi(params["z6"].str().c_str());
        unsigned char z7 = (params["z7"].str()=="{invalid}")?0:atoi(params["z7"].str().c_str());
        unsigned char z8 = (params["z8"].str()=="{invalid}")?0:atoi(params["z8"].str().c_str());
        setEZFloraProgram(id,program,z1,z2,z3,z4,z5,z6,z7,z8);
        json.addValue("ok","status");
    } else {
        json.addValue("Bad Program","status");
    }
}

void Insteon::ezflorastartprogram_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned char program = atoi(params["p"].str().c_str());
    if (program>0 && program<=4)
    {
        startProgram(id,program);
        json.addValue("ok","status");
    } else {
        json.addValue("Bad Program","status");
    }
}

void Insteon::ezflorastopprogram_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned char program = atoi(params["p"].str().c_str());
    if (program>0 && program<=4)
    {
        stopProgram(id,program);
        json.addValue("ok","status");
    } else {
        json.addValue("Bad Program","status");
    }
}

void Insteon::ezflorastartvalve_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned char valve = atoi(params["v"].str().c_str());
    if (valve>=0 && valve<=7)
    {
        startValve(id,valve);
        json.addValue("ok","status");
    } else {
        json.addValue("Bad valve number","status");
    }
}

void Insteon::ezflorastopvalve_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned char valve = atoi(params["v"].str().c_str());
    if (valve>=0 && valve<=7)
    {
        stopValve(id,valve);
        json.addValue("ok","status");
    } else {
        json.addValue("Bad valve number","status");
    }
}

void Insteon::ezfloraforcegetvalvestatus_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    mpInsteonModem->writeCommand(id,0x44,0x02);
    json.addValue("ok","status");
}

void Insteon::raw_callback(Dumais::JSON::JSON& json, Dumais::JSON::JSON& params)
{
    InsteonID id = strtoul(params["id"].str().c_str(),0,16);
    unsigned int cmd1 = strtoul(params["cmd1"].str().c_str(),0,16);
    unsigned int cmd2 = strtoul(params["cmd2"].str().c_str(),0,16);
    mpInsteonModem->writeCommand(id,cmd1,cmd2);
    json.addValue("ok","status");
}


void Insteon::onInsteonMessage(unsigned char *buf)
{
    InsteonID id = (buf[2]<<16)|(buf[3]<<8)|buf[4];
    // check if we know that device
    std::map<InsteonID,InsteonDevice*>::iterator it = mModules.find(id);
    if (it==mModules.end()) return;

    Dumais::JSON::JSON json;
    mModules[id]->onInsteonMessage(json,buf);
    if (json["event"].str()!="{invalid}") mpEventProcessor->processEvent(json);
}

void Insteon::onInsteonMessageSent(InsteonID id, IInsteonMessage* cmd)
{
    std::map<InsteonID,InsteonDevice*>::iterator it = mModules.find(id);
    if (it==mModules.end())
    {
        Logging::log("Insteon::onInsteonMessageSent ID not found");
        return;
    }
    
    StandardOrExtendedMessage *pMsg = dynamic_cast<StandardOrExtendedMessage*>(cmd);
    if (pMsg)
    {
        InsteonDirectMessage msg;
        msg.extended = pMsg->isExtended();
        msg.cmd1 = pMsg->command1();
        msg.cmd2 = pMsg->command2();
        if (msg.extended)
        {
            pMsg->copyData((char*)&msg.data[0]);
        }
        mModules[id]->setLastDirectMessageSent(msg);
    }
}

void Insteon::addModuleDefinition(InsteonID id, std::string name)
{
    SwitchDevice *dev = new SwitchDevice(name,id,mpInsteonModem);
    mModules[id] = dev;
}

void Insteon::addEZFlora(InsteonID id, std::string name)
{
    EZFloraDevice *dev = new EZFloraDevice(name,id,mpInsteonModem);
    mModules[id] = dev;
}


void Insteon::setController(InsteonID id)
{
    mControllerID = id;
}

void Insteon::clearModuleList()
{
    /* It is safe to delete all modules since only one service call at a time can be made in the system
     * This is guaranteed by the lock in the RESTInterface. Only the RESTInterface can make service calls.
    */
    this->suspend(); // suspend the Insteon thread so we dont access these devices
    for (std::map<InsteonID,InsteonDevice*>::iterator it=mModules.begin();it!=mModules.end();it++)
    {
        delete it->second;
    }
    mModules.clear();
    this->resume();
}

void Insteon::listModules(Dumais::JSON::JSON& json)
{

    for (std::map<InsteonID,InsteonDevice*>::iterator it=mModules.begin();it!=mModules.end();it++)
    {
        Dumais::JSON::JSON obj = json.addObject("module");
        it->second->toJSON(obj);
//        printf("TEST %s\r\n",obj.stringify(false).c_str());
    }
}

InsteonID Insteon::getControllerID()
{
    return mControllerID;
}


void Insteon::lightOn(InsteonID id,unsigned char level)
{

    // we want to avoid turning off the light using a light on command
    if (level<10) level = 10;

    unsigned char cmd = 0x11;
    if (level ==255)
    {
        cmd = 0x12;
    } else {
        cmd = 0x11;
    }
        
    //TODO: the "InsteonDevice" should send it 
    mpInsteonModem->writeCommand(id,cmd,level);
}

void Insteon::lightOff(InsteonID id)
{
    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeCommand(id,0x14,0);
}

void Insteon::lightToggle(InsteonID id,bool fast)
{
}

void Insteon::rampOn(InsteonID id, unsigned char level, unsigned char rate)
{
    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeCommand(id,0x2E,level<<4&(rate&0x0F));
}

void Insteon::rampOff(InsteonID id, unsigned char rate)
{
    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeCommand(id,0x2F,(rate&0x0F));
}


//Note: program 0 is default/manual timers. Programs are 1 to 4
void Insteon::setEZFloraProgram(InsteonID id, unsigned char program,unsigned char timer1,
            unsigned char timer2,unsigned char timer3,unsigned char timer4,unsigned char timer5,
            unsigned char timer6,unsigned char timer7,unsigned char timer8)

{
    unsigned char data[14];
    data[0] = timer1;
    data[1] = timer2;
    data[2] = timer3;
    data[3] = timer4;
    data[4] = timer5;
    data[5] = timer6;
    data[6] = timer7;
    data[7] = timer8;

    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeExtendedCommand(id,0x40,program,(unsigned char*)&data);
}

void Insteon::startProgram(InsteonID id, unsigned char program)
{
    if (program<1 || program >4) return;

    //TODO: the "InsteonDevice" should send it
    // clear meter counter
    mpInsteonModem->writeCommand(id,0x44,0x0F);

    // start program
    mpInsteonModem->writeCommand(id,0x42,program-1);
}

void Insteon::stopProgram(InsteonID id, unsigned char program)
{
    if (program<1 || program >4) return;

    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeCommand(id,0x43,program-1);
}

void Insteon::startValve(InsteonID id, unsigned char valve)
{
    if (valve>7) return;

    //TODO: the "InsteonDevice" should send it
    // clear meter counter
    mpInsteonModem->writeCommand(id,0x44,0x0F);

    // start valve
    mpInsteonModem->writeCommand(id,0x40,valve);
}

void Insteon::stopValve(InsteonID id, unsigned char valve)
{
    if (valve>7) return;

    //TODO: the "InsteonDevice" should send it
    mpInsteonModem->writeCommand(id,0x41,valve);
}


