#include "Container.h"
#include <unistd.h>
#include <iostream>
#include <sys/wait.h>
#include <sys/mount.h>
#include <sys/mman.h>
#include <signal.h>

#define STACKSIZE (1024*32)

Container::Container(strref path, Settings* s)
{
    this->cgroup = 0;
    this->chrootPath = path+"/chroot";
    this->settings = s;
    this->data = (SharedData*)mmap(0,sizeof(struct SharedData),PROT_READ|PROT_WRITE,MAP_SHARED|MAP_ANONYMOUS,-1,0);
    this->data->ready = false;

    mountPoints.push_back(new MountPoint("proc",this->chrootPath + "/proc","proc","",0));
    mountPoints.push_back(new MountPoint("sys",this->chrootPath + "/sys","sysfs","",0));
    mountPoints.push_back(new MountPoint("/dev",this->chrootPath + "/dev","none","",MS_BIND));
}

void Container::start()
{
    if (make_cgroup()) return;
    if (configure_cgroup()) return;
    start_process();
}

void Container::clean()
{
    for (auto& it : mountPoints)
    {   
        it->umount();
        delete it;
    }
    mountPoints.clear();

    delete this->veth;
    if (this->cgroup) delete cgroup;
    this->cgroup = 0;
    munmap(this->data,sizeof(struct SharedData));
}

int Container::make_chroot()
{
    for (auto& it : mountPoints)
    {
        if (!it->mount()) return -1;
    }

    if (chdir(this->chrootPath.c_str())) return -1;
    if (chroot(this->chrootPath.c_str())) return -1;

    return 0;
}

int Container::make_cgroup()
{
    unsigned long id = getpid();
    std::string name = "awesome-"+std::to_string(id);
    cgroup = new CGroup(name);
    if (!cgroup->is_valid()) return -1;
    return 0;
}

int Container::configure_cgroup()
{
    std::string memlimit = settings->get("memlimit","100000000");
    // Set a limit of 4mb for memory footprint
    std::cout << "Mem limit: " << memlimit << " bytes\r\n"; 
    cgroup->write_file("memory","memory.limit_in_bytes",memlimit);


    std::string cpupercent = settings->get("cpupercent","100");
    double d = std::round((std::stod(cpupercent) / 100.0)*1024.0);
    std::string val = std::to_string((int)d);
    std::cout << "CPU shares: " << val << " (" << cpupercent <<"%)\r\n";
    cgroup->write_file("cpu","cpu.shares",val);

    return 0;
}

int new_process(void* data)
{
    Container* c = (Container*)data;
    c->run_child();
}

void Container::run_child()
{
    Veth* veth = new Veth(this->id);

    while (!this->data->ready);
    std::string app = this->settings->get("process","/bin/sh");
    std::string ip = this->settings->get("ip","169.254.0.1");
    std::string gw = this->settings->get("gw","169.254.0.2");

    if (make_chroot())
    {
        std::cerr << "Could not chroot into " << this->chrootPath << "\r\n";
        exit(0);
    }


    std::string hostname = "container";
    sethostname(hostname.c_str(),hostname.size());
    veth->up(Veth::Guest);
    veth->set_ip(Veth::Guest, ip);
    veth->setgw(gw);

    // Drop privileges
    std::string uid = settings->get("user","0");
    std::string gid = settings->get("group","0");
    std::cout << "Dropping privileges to " << uid << ":" << gid << "\r\n";
    setgid(std::stoi(gid));
    setuid(std::stoi(uid));

    // get arguments
    std::string arguments[128];
    char* args[128] = {0};
    arguments[0] = app;
    size_t index = 1;

    std::cout << "Starting " << app;
    while (index < 128)
    {
        arguments[index] = settings->get("arg"+std::to_string(index),"");
        if (arguments[index] == "") break;
        std::cout << " " << arguments[index];
        index++;
    }
    for (int i = 0; i < 128; i++)
    {
        args[i] = 0;
        if (arguments[i] != "") args[i] = const_cast<char*>(arguments[i].c_str());
    }

    std::cout << "\r\n\r\n";
    if (execv(app.c_str(),(char**)&args) != 0)
    {
        std::cerr << "Could not load " << app << "\r\n";
        exit(0);
    }
    // exec will never return, so we don't have to take care of this.
}

int Container::start_process()
{
    char* stack = new char[STACKSIZE];

    std::string id = std::to_string(getpid());
    this->id = id;

    unsigned long child = clone(new_process,(char*)&stack[STACKSIZE],CLONE_NEWPID|CLONE_NEWNS|CLONE_NEWNET|CLONE_NEWUTS|SIGCHLD,this);


    this->veth = new Veth();
    this->veth->create(id, child);
    this->veth->up(Veth::Host);
    std::string bridge = this->settings->get("bridge","");
    if (bridge != "") this->veth->add_to_bridge(bridge);

    // Set that child process as the root process of the cgroup tree
    cgroup->add_task(child);
    std::cout << "Added PID " << child << " in cgroup\r\n";

    this->data->ready = true;

    // wait for the child to return and we will then cleanup
    while (1)
    {
        int status;
        if (waitpid(child,&status,0) == -1) break;
        if (WIFEXITED(status))
        {
            break;
        }
    }

    if (bridge != "") this->veth->remove_from_bridge(bridge);

    this->veth->destroy();
    delete[] stack;
    this->clean();
    return 0;
}

