/*
 * Asterisk -- An open source telephony toolkit.
 *
 * Copyright (C) <2009>, <Patrick Dumais>
 *
 * <Patrick Dumais> <<asterisk@dumaisnet.ca>>
 *
 * See http://www.asterisk.org for more information about
 * the Asterisk project. Please do not directly contact
 * any of the maintainers of this project for assistance;
 * the project provides a web site, mailing lists and IRC
 * channels for your use.
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License Version 2. See the LICENSE file
 * at the top of the source tree.
 */
/*** MODULEINFO
	<defaultenabled>no</defaultenabled>
 ***/

#ifndef AST_MODULE
#define AST_MODULE "app_inject"
#endif
#include "asterisk.h"

ASTERISK_FILE_VERSION(__FILE__, "$Revision: 40722 $")

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "asterisk/file.h"
#include "asterisk/logger.h"
#include "asterisk/channel.h"
#include "asterisk/pbx.h"
#include "asterisk/module.h"
#include "asterisk/lock.h"
#include "asterisk/app.h"
#include "asterisk/audiohook.h"

static char *app = "Inject";
static char *synopsis = "Inject.";
static char *descrip = "Inject\n";

static int stopPlayFile(struct ast_channel *chan);
static int generate_injection(struct ast_channel *chan, struct ast_audiohook *hook);


static struct injectinfo
{
	int active;
	struct ast_audiohook *hook;	
	struct ast_channel *chan;
};

static void *thread1(void *obj)
{
	struct injectinfo* info = obj;

	while (info->active && info->hook->status == AST_AUDIOHOOK_STATUS_RUNNING){
		ast_audiohook_trigger_wait(info->hook);
		generate_injection(info->chan, info->hook);
	}

	return 0;
}


static void *spy_alloc(struct ast_channel *chan, void *data)
{
        return data;
}

static void spy_release(struct ast_channel *chan, void *data)
{
}

static int spy_generate(struct ast_channel *chan, void *data, int len, int samples)
{

        struct ast_frame *f;
	struct ast_audiohook *hook = data;

        ast_audiohook_lock(hook);

        f = ast_audiohook_read_frame(hook, samples, AST_AUDIOHOOK_DIRECTION_BOTH, AST_FORMAT_SLINEAR);

        ast_audiohook_unlock(hook);

        if (!f) return 0;

        if (ast_write(chan, f)) {
                ast_frfree(f);
                return -1;
        }

        ast_frfree(f);

        return 0;

}


static int generate_injection(struct ast_channel *chan, struct ast_audiohook *hook)
{

        struct ast_frame *f = 0;

	// Is a file stream opened?
	if (chan->stream){
		f = ast_readframe(chan->stream);
		if (!f){
			stopPlayFile(chan);
			return 0;
		}
	} else {
		return 0;
	}



        ast_audiohook_lock(hook);
	ast_audiohook_write_frame(hook, AST_AUDIOHOOK_DIRECTION_WRITE, f);
        ast_audiohook_unlock(hook);


        ast_frfree(f);

        return 0;

}


static int startPlayFile(struct ast_channel *chan, char* fname)
{
	// last parameter set to 1 to indicate that we don't wanna stop the generator
	ast_verbose(VERBOSE_PREFIX_2 "Playing %s\n",fname);
	ast_openstream_full(chan, fname, chan->language, 1);
}

static int stopPlayFile(struct ast_channel *chan)
{
	ast_closestream(chan->stream);
        chan->stream = NULL;
}


static int app_exec(struct ast_channel *chan, void *data)
{
	struct ast_flags flags;
	struct ast_module_user *u;
	struct ast_config *cfg;


	struct ast_generator spygen1;
	spygen1.alloc = spy_alloc;
	spygen1.release = spy_release;
	spygen1.generate = spy_generate;


	// Parsing arguments
	AST_DECLARE_APP_ARGS(args,
                AST_APP_ARG(channame);
        );

	u = ast_module_user_add(chan);
	void* parse = ast_strdupa(data);
	AST_STANDARD_APP_ARGS(args, parse);


	// Answer channel if not already answered
	if (chan->_state != AST_STATE_UP) ast_answer(chan);


	// Get an instance of the channel passed in parameters
	struct ast_channel *chan2 = 0;
	chan2 = ast_get_channel_by_name_prefix_locked(args.channame,strlen(args.channame));


	// The meat starts here
	if (!chan2) {
                ast_log(LOG_WARNING, "No such channel: %s\n", args.channame);
		
        } else {
		ast_channel_unlock(chan2);
		struct ast_channel *peer =0;
		peer = ast_bridged_channel(chan2);	
		if (peer){
			ast_log(LOG_NOTICE, "Found call involving %s and %s\n", chan2->name,peer->name);
			struct ast_audiohook hook1;
			memset(&hook1,0,sizeof(hook1));
			ast_audiohook_init(&hook1,AST_AUDIOHOOK_TYPE_SPY,"Inject");
			ast_audiohook_attach(chan2, &hook1);
			ast_activate_generator(chan, &spygen1, &hook1);

			struct ast_audiohook hook2;
                        memset(&hook2,0,sizeof(hook2));
                        ast_audiohook_init(&hook2,AST_AUDIOHOOK_TYPE_WHISPER,"Inject");
                        ast_audiohook_attach(chan2, &hook2);

			struct injectinfo info;
			info.hook = &hook2;
			info.active = 1;
			info.chan = chan2;

			ast_set_flag(&hook2, AST_AUDIOHOOK_TRIGGER_WRITE);
			pthread_attr_t attr;
		        pthread_t thread;
			pthread_attr_init(&attr);
			pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
			ast_pthread_create_background(&thread, &attr, thread1, &info);
			pthread_attr_destroy(&attr);



			ast_set_read_format(chan, AST_FORMAT_SLINEAR);
			ast_set_write_format(chan, AST_FORMAT_SLINEAR);
			int error = 1;
			struct ast_frame *f;
			int ms;
        		while (1) {
				// Wait 100ms, because we wanna check if someone hung-up
				ms = 100;
				struct ast_channel *c = ast_waitfor_n(&chan, 1, &ms);
				if (ast_check_hangup(peer))
				{
					ast_verbose(VERBOSE_PREFIX_2 "Explicit disconnection\n");
					break;
				}
				if (ast_check_hangup(chan))
                                {
                                        ast_verbose(VERBOSE_PREFIX_2 "Explicit disconnection\n");
                                        break;
                                }
				if (ast_check_hangup(chan2))
                                {
                                        ast_verbose(VERBOSE_PREFIX_2 "Implicit disconnection\n");
                                        break;
                                }

				if (c==0) continue;

	        	        f = ast_read(chan);
                		if (!f) break;

		                f->delivery.tv_sec = 0;
        		        f->delivery.tv_usec = 0;

	        	        if (f->frametype == AST_FRAME_DTMF){
	                	        char num = f->subclass;
					if (num == '#'){
						break;				
					} else if (num >= '0' && num <= '9'){
						char fn[256];
						sprintf((char*)&fn,"injection%c",num);
						startPlayFile(chan2,(char*)&fn);
					}
				} else if (f->frametype == AST_FRAME_VOICE){
				}
	

				ast_write(chan,f);
				ast_frame_free(f, 0);
		
			}

			info.active = 0;
                        ast_audiohook_detach(&hook2);
                        ast_audiohook_destroy(&hook2);

			ast_deactivate_generator(chan);
			ast_audiohook_detach(&hook1);
			ast_audiohook_destroy(&hook1);
			pthread_join(thread1, 0);
			
		}

	}


	// Clean-up
	ast_verbose(VERBOSE_PREFIX_2 "Exiting\n");
	ast_module_user_remove(u);
	return 0;
}


static int unload_module(void)
{
	int res;
	res = ast_unregister_application(app);
	return res;	
}

static int load_module(void)
{
	return ast_register_application(app, app_exec, synopsis, descrip);
}

AST_MODULE_INFO_STANDARD(ASTERISK_GPL_KEY, "app_inject");
